<?php
// functions.php
function h($s){ return htmlspecialchars($s, ENT_QUOTES, 'UTF-8'); }

function get_client_ip(){
    if (!empty($_SERVER['HTTP_CLIENT_IP'])) return $_SERVER['HTTP_CLIENT_IP'];
    if (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) return explode(',', $_SERVER['HTTP_X_FORWARDED_FOR'])[0];
    return $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
}

function make_thumbnail($srcPath, $destPath, $maxW = 300, $maxH = 300){
    $ext = strtolower(pathinfo($srcPath, PATHINFO_EXTENSION));
    if (!in_array($ext, ['jpg','jpeg','png','gif'])) return false;
    switch ($ext) {
        case 'jpg':
        case 'jpeg':
            $img = imagecreatefromjpeg($srcPath); break;
        case 'png':
            $img = imagecreatefrompng($srcPath); break;
        case 'gif':
            $img = imagecreatefromgif($srcPath); break;
        default:
            return false;
    }
    if (!$img) return false;
    $w = imagesx($img);
    $h = imagesy($img);
    $ratio = min($maxW/$w, $maxH/$h);
    $nw = max(1, floor($w * $ratio));
    $nh = max(1, floor($h * $ratio));
    $tmp = imagecreatetruecolor($nw, $nh);
    // preserve PNG transparency
    if ($ext === 'png') {
        imagealphablending($tmp, false);
        imagesavealpha($tmp, true);
    }
    imagecopyresampled($tmp, $img, 0,0,0,0, $nw, $nh, $w, $h);
    $ok = imagejpeg($tmp, $destPath, 85);
    imagedestroy($img);
    imagedestroy($tmp);
    return $ok;
}

function generate_thumbnail(string $srcPath, string $destPath, int $maxW = 1200, int $maxH = 800, int $quality = 90): bool {
    // jika sumber tidak ada
    if (!file_exists($srcPath)) return false;

    $info = @getimagesize($srcPath);
    $mime = $info['mime'] ?? '';

    if ($mime === 'application/pdf') {
        if (class_exists('Imagick')) {
            try {
                $im = new \Imagick();
                $im->setResolution(150, 150); // set resolusi render => lebih tajam
                $im->readImage($srcPath . '[0]');
                $im->setImageBackgroundColor('white');
                $im = $im->flattenImages();
                $im->setImageColorspace(\Imagick::COLORSPACE_RGB);
                $im->setImageFormat('jpeg');
                $im->resizeImage($maxW, $maxH, \Imagick::FILTER_LANCZOS, 1, true);
                $im->setImageCompressionQuality($quality);
                $im->writeImage($destPath);
                $im->clear();
                $im->destroy();
                return true;
            } catch (Exception $e) {
                return false;
            }
        } else {
            return false;
        }
    }

    switch ($mime) {
        case 'image/jpeg':
        case 'image/jpg':
            $srcImg = imagecreatefromjpeg($srcPath);
            break;
        case 'image/png':
            $srcImg = imagecreatefrompng($srcPath);
            break;
        case 'image/gif':
            $srcImg = imagecreatefromgif($srcPath);
            break;
        default:
            return false;
    }
    if (!$srcImg) return false;

    $w = imagesx($srcImg);
    $h = imagesy($srcImg);

    if (($mime === 'image/jpeg' || $mime === 'image/jpg') && function_exists('exif_read_data')) {
        try {
            $exif = @exif_read_data($srcPath);
            if (!empty($exif['Orientation'])) {
                $ort = $exif['Orientation'];
                if ($ort == 3) $srcImg = imagerotate($srcImg, 180, 0);
                elseif ($ort == 6) $srcImg = imagerotate($srcImg, -90, 0);
                elseif ($ort == 8) $srcImg = imagerotate($srcImg, 90, 0);
                // recompute sizes after rotate
                $w = imagesx($srcImg);
                $h = imagesy($srcImg);
            }
        } catch (Exception $e) { /* ignore exif errors */ }
    }

    $ratio = min($maxW / $w, $maxH / $h, 1); // jangan perbesar jika sumber lebih kecil
    $newW = (int) max(1, floor($w * $ratio));
    $newH = (int) max(1, floor($h * $ratio));

    $thumb = imagecreatetruecolor($newW, $newH);

    if ($mime === 'image/png' || $mime === 'image/gif') {
        imagealphablending($thumb, false);
        imagesavealpha($thumb, true);
        $transparent = imagecolorallocatealpha($thumb, 0, 0, 0, 127);
        imagefilledrectangle($thumb, 0, 0, $newW, $newH, $transparent);
    } else {
        $white = imagecolorallocate($thumb, 255, 255, 255);
        imagefilledrectangle($thumb, 0, 0, $newW, $newH, $white);
    }

    imagecopyresampled($thumb, $srcImg, 0, 0, 0, 0, $newW, $newH, $w, $h);

    $saved = imagejpeg($thumb, $destPath, $quality);

    imagedestroy($thumb);
    imagedestroy($srcImg);

    return $saved;
}
